/*
 * Decompiled with CFR 0.152.
 */
package haveno.common.persistence;

import com.google.inject.Inject;
import com.google.inject.name.Named;
import haveno.common.Timer;
import haveno.common.UserThread;
import haveno.common.app.DevEnv;
import haveno.common.crypto.CryptoException;
import haveno.common.crypto.Encryption;
import haveno.common.crypto.KeyRing;
import haveno.common.file.CorruptedStorageFileHandler;
import haveno.common.file.FileUtil;
import haveno.common.handlers.ResultHandler;
import haveno.common.proto.persistable.PersistableEnvelope;
import haveno.common.proto.persistable.PersistenceProtoResolver;
import haveno.common.util.GcUtil;
import haveno.common.util.Preconditions;
import haveno.common.util.SingleThreadExecutorUtils;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PersistenceManager<T extends PersistableEnvelope> {
    private static final Logger log = LoggerFactory.getLogger(PersistenceManager.class);
    public static final Map<String, PersistenceManager<?>> ALL_PERSISTENCE_MANAGERS = new HashMap();
    private static boolean flushAtShutdownCalled;
    private static final AtomicBoolean allServicesInitialized;
    private final File dir;
    private final PersistenceProtoResolver persistenceProtoResolver;
    private final CorruptedStorageFileHandler corruptedStorageFileHandler;
    @Nullable
    private final KeyRing keyRing;
    private File storageFile;
    private T persistable;
    private String fileName;
    private Source source = Source.PRIVATE_LOW_PRIO;
    private Path usedTempFilePath;
    private volatile boolean persistenceRequested;
    @Nullable
    private Timer timer;
    private ExecutorService writeToDiskExecutor;
    public final AtomicBoolean initCalled = new AtomicBoolean(false);
    public final AtomicBoolean readCalled = new AtomicBoolean(false);

    public static void onAllServicesInitialized() {
        allServicesInitialized.set(true);
        ALL_PERSISTENCE_MANAGERS.values().forEach(persistenceManager -> {
            if (persistenceManager.persistenceRequested) {
                persistenceManager.maybeStartTimerForPersistence();
            }
        });
    }

    public static void flushAllDataToDiskAtBackup(ResultHandler completeHandler) {
        PersistenceManager.flushAllDataToDisk(completeHandler, false);
    }

    public static void flushAllDataToDiskAtShutdown(ResultHandler completeHandler) {
        PersistenceManager.flushAllDataToDisk(completeHandler, true);
    }

    public static void reset() {
        ALL_PERSISTENCE_MANAGERS.clear();
        flushAtShutdownCalled = false;
        allServicesInitialized.set(false);
    }

    private static void flushAllDataToDisk(ResultHandler completeHandler, boolean doShutdown) {
        if (!allServicesInitialized.get()) {
            log.warn("Application has not completed start up yet so we do not flush data to disk.");
            completeHandler.handleResult();
            return;
        }
        UserThread.execute(() -> {
            if (doShutdown) {
                if (flushAtShutdownCalled) {
                    log.warn("We got flushAllDataToDisk called again. This can happen in some rare cases. We ignore the repeated call.");
                    return;
                }
                flushAtShutdownCalled = true;
            }
            log.info("Start flushAllDataToDisk");
            AtomicInteger openInstances = new AtomicInteger(ALL_PERSISTENCE_MANAGERS.size());
            if (openInstances.get() == 0) {
                log.info("No PersistenceManager instances have been created yet.");
                completeHandler.handleResult();
            }
            new HashSet(ALL_PERSISTENCE_MANAGERS.values()).forEach(persistenceManager -> {
                if (persistenceManager.readCalled.get() && (persistenceManager.source.flushAtShutDown || persistenceManager.persistenceRequested)) {
                    try {
                        persistenceManager.persistNow(() -> UserThread.execute(() -> PersistenceManager.onWriteCompleted(completeHandler, openInstances, persistenceManager, doShutdown)));
                    }
                    catch (Exception e) {
                        if (!doShutdown) {
                            throw e;
                        }
                        log.warn("Error flushing data to disk on shut down. Calling completeHandler.");
                        UserThread.execute(() -> PersistenceManager.onWriteCompleted(completeHandler, openInstances, persistenceManager, doShutdown));
                    }
                } else {
                    PersistenceManager.onWriteCompleted(completeHandler, openInstances, persistenceManager, doShutdown);
                }
            });
        });
    }

    private static void onWriteCompleted(ResultHandler completeHandler, AtomicInteger openInstances, PersistenceManager<?> persistenceManager, boolean doShutdown) {
        if (doShutdown) {
            persistenceManager.shutdown();
        }
        if (openInstances.decrementAndGet() == 0) {
            log.info("flushAllDataToDisk completed");
            completeHandler.handleResult();
        }
    }

    @Inject
    public PersistenceManager(@Named(value="storageDir") File dir, PersistenceProtoResolver persistenceProtoResolver, CorruptedStorageFileHandler corruptedStorageFileHandler, @Nullable KeyRing keyRing) {
        this.dir = Preconditions.checkDir(dir);
        this.persistenceProtoResolver = persistenceProtoResolver;
        this.corruptedStorageFileHandler = corruptedStorageFileHandler;
        this.keyRing = keyRing;
    }

    public void initialize(T persistable, Source source) {
        this.initialize(persistable, persistable.getDefaultStorageFileName(), source);
    }

    public void initialize(T persistable, String fileName, Source source) {
        if (flushAtShutdownCalled) {
            log.warn("We have started the shut down routine already. We ignore that initialize call.");
            return;
        }
        if (ALL_PERSISTENCE_MANAGERS.containsKey(fileName)) {
            RuntimeException runtimeException = new RuntimeException("We must not create multiple PersistenceManager instances for file " + fileName + ".");
            runtimeException.printStackTrace();
            throw runtimeException;
        }
        if (this.initCalled.get()) {
            RuntimeException runtimeException = new RuntimeException("We must not call initialize multiple times. PersistenceManager for file: " + fileName + ".");
            runtimeException.printStackTrace();
            throw runtimeException;
        }
        this.initCalled.set(true);
        this.persistable = persistable;
        this.fileName = fileName;
        this.source = source;
        this.storageFile = new File(this.dir, fileName);
        ALL_PERSISTENCE_MANAGERS.put(fileName, this);
    }

    public void shutdown() {
        ALL_PERSISTENCE_MANAGERS.remove(this.fileName);
        if (this.timer != null) {
            this.timer.stop();
        }
        if (this.writeToDiskExecutor != null) {
            this.writeToDiskExecutor.shutdown();
        }
    }

    public void readPersisted(Consumer<T> resultHandler, Runnable orElse) {
        this.readPersisted((String)com.google.common.base.Preconditions.checkNotNull((Object)this.fileName), resultHandler, orElse);
    }

    public void readPersisted(String fileName, Consumer<T> resultHandler, Runnable orElse) {
        if (flushAtShutdownCalled) {
            log.warn("We have started the shut down routine already. We ignore that readPersisted call.");
            return;
        }
        new Thread(() -> {
            Object persisted = this.getPersisted(fileName);
            if (persisted != null) {
                UserThread.execute(() -> {
                    resultHandler.accept(persisted);
                    GcUtil.maybeReleaseMemory();
                });
            } else {
                UserThread.execute(orElse);
            }
        }, "PersistenceManager-read-" + fileName).start();
    }

    @Nullable
    public T getPersisted() {
        return this.getPersisted((String)com.google.common.base.Preconditions.checkNotNull((Object)this.fileName));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Nullable
    public T getPersisted(String fileName) {
        if (flushAtShutdownCalled) {
            log.warn("We have started the shut down routine already. We ignore that getPersisted call.");
            return null;
        }
        if (this.keyRing != null && !this.keyRing.isUnlocked()) {
            log.warn("Account is not open yet, ignoring getPersisted.");
            return null;
        }
        this.readCalled.set(true);
        File storageFile = new File(this.dir, fileName);
        if (!storageFile.exists()) {
            return null;
        }
        long ts = System.currentTimeMillis();
        try (FileInputStream fileInputStream = new FileInputStream(storageFile);){
            protobuf.PersistableEnvelope proto222;
            if (this.keyRing != null) {
                protobuf.PersistableEnvelope proto;
                byte[] encryptedBytes = fileInputStream.readAllBytes();
                try {
                    byte[] decryptedBytes = Encryption.decryptPayloadWithHmac(encryptedBytes, this.keyRing.getSymmetricKey());
                    proto = protobuf.PersistableEnvelope.parseFrom((byte[])decryptedBytes);
                }
                catch (CryptoException ce) {
                    log.warn("Expected encrypted persisted file, attempting to getPersisted without decryption");
                    ByteArrayInputStream bs = new ByteArrayInputStream(encryptedBytes);
                    proto = protobuf.PersistableEnvelope.parseDelimitedFrom((InputStream)bs);
                }
            } else {
                proto222 = protobuf.PersistableEnvelope.parseDelimitedFrom((InputStream)fileInputStream);
            }
            PersistableEnvelope persistableEnvelope = this.persistenceProtoResolver.fromProto(proto222);
            log.info("Reading {} completed in {} ms", (Object)fileName, (Object)(System.currentTimeMillis() - ts));
            PersistableEnvelope persistableEnvelope2 = persistableEnvelope;
            return (T)persistableEnvelope2;
        }
        catch (Throwable t) {
            log.error("Reading {} failed with {}.", (Object)fileName, (Object)t.getMessage());
            try {
                FileUtil.removeAndBackupFile(this.dir, storageFile, fileName, "backup_of_corrupted_data");
                DevEnv.logErrorAndThrowIfDevMode(t.toString());
            }
            catch (IOException e1) {
                e1.printStackTrace();
                log.error(e1.getMessage());
            }
            if (this.corruptedStorageFileHandler == null) return null;
            this.corruptedStorageFileHandler.addFile(storageFile.getName());
            return null;
        }
    }

    public void requestPersistence() {
        if (flushAtShutdownCalled) {
            log.warn("We have started the shut down routine already. We ignore that requestPersistence call.");
            try {
                throw new RuntimeException("We have started the shut down routine already. We ignore that requestPersistence call.");
            }
            catch (Exception e) {
                e.printStackTrace();
                return;
            }
        }
        this.persistenceRequested = true;
        if (!allServicesInitialized.get()) {
            return;
        }
        this.maybeStartTimerForPersistence();
    }

    private void maybeStartTimerForPersistence() {
        UserThread.execute(() -> {
            if (this.timer == null) {
                this.timer = UserThread.runAfter(() -> {
                    this.persistNow(null);
                    UserThread.execute(() -> {
                        this.timer = null;
                    });
                }, this.source.delay, TimeUnit.MILLISECONDS);
            }
        });
    }

    public void forcePersistNow() {
        this.persistNow(null, true);
    }

    public void persistNow(@Nullable Runnable completeHandler) {
        this.persistNow(completeHandler, false);
    }

    private synchronized void persistNow(@Nullable Runnable completeHandler, boolean force) {
        long ts = System.currentTimeMillis();
        try {
            protobuf.PersistableEnvelope serialized = (protobuf.PersistableEnvelope)this.persistable.toPersistableMessage();
            this.getWriteToDiskExecutor().execute(() -> this.writeToDisk(serialized, completeHandler, force));
            long duration = System.currentTimeMillis() - ts;
            if (duration > 100L) {
                log.info("Serializing {} took {} msec", (Object)this.fileName, (Object)duration);
            }
        }
        catch (Throwable e) {
            log.error("Error in saveToFile toProtoMessage: {}, {}", (Object)this.persistable.getClass().getSimpleName(), (Object)this.fileName);
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeToDisk(protobuf.PersistableEnvelope serialized, @Nullable Runnable completeHandler, boolean force) {
        if (!allServicesInitialized.get() && !force) {
            log.warn("Application has not completed start up yet so we do not permit writing data to disk.");
            if (completeHandler != null) {
                UserThread.execute(completeHandler);
            }
            return;
        }
        if (this.keyRing != null && !this.keyRing.isUnlocked()) {
            log.warn("Account is not open, ignoring writeToDisk.");
            if (completeHandler != null) {
                UserThread.execute(completeHandler);
            }
            return;
        }
        long ts = System.currentTimeMillis();
        File tempFile = null;
        FileOutputStream fileOutputStream = null;
        try {
            FileUtil.rollingBackup(this.dir, this.fileName, this.source.getNumMaxBackupFiles());
            if (!this.dir.exists() && !this.dir.mkdir()) {
                log.warn("make dir failed {}", (Object)this.fileName);
            }
            tempFile = this.usedTempFilePath != null ? FileUtil.createNewFile(this.usedTempFilePath) : File.createTempFile("temp_" + this.fileName, null, this.dir);
            tempFile.deleteOnExit();
            fileOutputStream = new FileOutputStream(tempFile);
            if (this.keyRing != null) {
                byte[] encryptedBytes = Encryption.encryptPayloadWithHmac(serialized.toByteArray(), this.keyRing.getSymmetricKey());
                fileOutputStream.write(encryptedBytes);
            } else {
                serialized.writeDelimitedTo((OutputStream)fileOutputStream);
            }
            fileOutputStream.flush();
            fileOutputStream.getFD().sync();
            fileOutputStream.close();
            FileUtil.renameFile(tempFile, this.storageFile);
            this.usedTempFilePath = tempFile.toPath();
        }
        catch (Throwable t) {
            this.usedTempFilePath = null;
            log.error("Error at saveToFile, storageFile={}", (Object)this.fileName, (Object)t);
        }
        finally {
            if (tempFile != null && tempFile.exists()) {
                log.warn("Temp file still exists after failed save. We will delete it now. storageFile={}", (Object)this.fileName);
                if (!tempFile.delete()) {
                    log.error("Cannot delete temp file.");
                }
            }
            try {
                if (fileOutputStream != null) {
                    fileOutputStream.close();
                }
            }
            catch (IOException e) {
                e.printStackTrace();
                log.error("Cannot close resources." + e.getMessage());
            }
            long duration = System.currentTimeMillis() - ts;
            if (duration > 100L) {
                log.info("Writing the serialized {} completed in {} msec", (Object)this.fileName, (Object)duration);
            }
            this.persistenceRequested = false;
            if (completeHandler != null) {
                UserThread.execute(completeHandler);
            }
        }
    }

    private ExecutorService getWriteToDiskExecutor() {
        if (this.writeToDiskExecutor == null) {
            String name = "Write-" + this.fileName + "_to-disk";
            this.writeToDiskExecutor = SingleThreadExecutorUtils.getSingleThreadExecutor(name);
        }
        return this.writeToDiskExecutor;
    }

    public String toString() {
        return "PersistenceManager{\n     fileName='" + this.fileName + "',\n     dir=" + String.valueOf(this.dir) + ",\n     storageFile=" + String.valueOf(this.storageFile) + ",\n     persistable=" + String.valueOf(this.persistable) + ",\n     source=" + String.valueOf((Object)this.source) + ",\n     usedTempFilePath=" + String.valueOf(this.usedTempFilePath) + ",\n     persistenceRequested=" + this.persistenceRequested + "\n}";
    }

    static {
        allServicesInitialized = new AtomicBoolean(false);
    }

    public static enum Source {
        NETWORK(1, TimeUnit.MINUTES.toMillis(5L), false),
        PRIVATE(10, 200L, true),
        PRIVATE_LOW_PRIO(4, TimeUnit.MINUTES.toMillis(1L), false);

        private final int numMaxBackupFiles;
        private final long delay;
        private final boolean flushAtShutDown;

        private Source(int numMaxBackupFiles, long delay, boolean flushAtShutDown) {
            this.numMaxBackupFiles = numMaxBackupFiles;
            this.delay = delay;
            this.flushAtShutDown = flushAtShutDown;
        }

        public int getNumMaxBackupFiles() {
            return this.numMaxBackupFiles;
        }

        public long getDelay() {
            return this.delay;
        }

        public boolean isFlushAtShutDown() {
            return this.flushAtShutDown;
        }
    }
}

