/*
 * Decompiled with CFR 0.152.
 */
package haveno.common.crypto;

import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.google.inject.name.Named;
import haveno.common.crypto.CryptoException;
import haveno.common.crypto.Encryption;
import haveno.common.crypto.IncorrectPasswordException;
import haveno.common.crypto.KeyRing;
import haveno.common.file.FileUtil;
import haveno.common.util.Preconditions;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.UnrecoverableKeyException;
import java.security.interfaces.DSAParams;
import java.security.interfaces.DSAPrivateKey;
import java.security.interfaces.RSAPrivateCrtKey;
import java.security.spec.DSAPublicKeySpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.RSAPublicKeySpec;
import javax.crypto.SecretKey;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class KeyStorage {
    private static final Logger log = LoggerFactory.getLogger(KeyStorage.class);
    private final File storageDir;

    @Inject
    public KeyStorage(@Named(value="keyStorageDir") File storageDir) {
        this.storageDir = Preconditions.checkDir(storageDir);
    }

    public boolean allKeyFilesExist() {
        return this.fileExists(KeyEntry.MSG_SIGNATURE) && this.fileExists(KeyEntry.MSG_ENCRYPTION) && this.fileExists(KeyEntry.SYM_ENCRYPTION);
    }

    private boolean fileExists(KeyEntry keyEntry) {
        return new File(String.valueOf(this.storageDir) + "/" + keyEntry.getFileName()).exists();
    }

    private byte[] loadKeyBytes(KeyEntry keyEntry, SecretKey secretKey) {
        byte[] byArray;
        File keyFile = new File(String.valueOf(this.storageDir) + "/" + keyEntry.getFileName());
        FileInputStream fis = new FileInputStream(keyFile.getPath());
        try {
            byte[] encodedKey = new byte[(int)keyFile.length()];
            fis.read(encodedKey);
            byArray = encodedKey = Encryption.decryptPayloadWithHmac(encodedKey, secretKey);
        }
        catch (Throwable throwable) {
            try {
                try {
                    fis.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (CryptoException | IOException e) {
                log.error("Could not load key " + keyEntry.toString(), (Object)e.getMessage());
                throw new RuntimeException("Could not load key " + keyEntry.toString(), e);
            }
        }
        fis.close();
        return byArray;
    }

    public KeyPair loadKeyPair(KeyEntry keyEntry, SecretKey secretKey) {
        FileUtil.rollingBackup(this.storageDir, keyEntry.getFileName() + ".key", 20);
        try {
            PublicKey publicKey;
            KeyFactory keyFactory = KeyFactory.getInstance(keyEntry.getAlgorithm());
            byte[] encodedPrivateKey = this.loadKeyBytes(keyEntry, secretKey);
            PKCS8EncodedKeySpec privateKeySpec = new PKCS8EncodedKeySpec(encodedPrivateKey);
            PrivateKey privateKey = keyFactory.generatePrivate(privateKeySpec);
            if (privateKey instanceof RSAPrivateCrtKey) {
                RSAPrivateCrtKey rsaPrivateKey = (RSAPrivateCrtKey)privateKey;
                RSAPublicKeySpec publicKeySpec = new RSAPublicKeySpec(rsaPrivateKey.getModulus(), rsaPrivateKey.getPublicExponent());
                publicKey = keyFactory.generatePublic(publicKeySpec);
            } else if (privateKey instanceof DSAPrivateKey) {
                DSAPrivateKey dsaPrivateKey = (DSAPrivateKey)privateKey;
                DSAParams dsaParams = dsaPrivateKey.getParams();
                BigInteger p = dsaParams.getP();
                BigInteger q = dsaParams.getQ();
                BigInteger g = dsaParams.getG();
                BigInteger y = g.modPow(dsaPrivateKey.getX(), p);
                DSAPublicKeySpec publicKeySpec = new DSAPublicKeySpec(y, p, q, g);
                publicKey = keyFactory.generatePublic(publicKeySpec);
            } else {
                throw new RuntimeException("Unsupported key algo" + keyEntry.getAlgorithm());
            }
            return new KeyPair(publicKey, privateKey);
        }
        catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            log.error("Could not load key " + keyEntry.toString(), (Throwable)e);
            throw new RuntimeException("Could not load key " + keyEntry.toString(), e);
        }
    }

    public SecretKey loadSecretKey(KeyEntry keyEntry, String password) throws IncorrectPasswordException {
        char[] passwordChars = password == null ? new char[]{} : password.toCharArray();
        try {
            KeyStore keyStore = KeyStore.getInstance("PKCS12");
            keyStore.load(new FileInputStream(String.valueOf(this.storageDir) + "/" + keyEntry.getFileName()), passwordChars);
            Key key = keyStore.getKey(keyEntry.getAlias(), passwordChars);
            return (SecretKey)key;
        }
        catch (UnrecoverableKeyException e) {
            throw new IncorrectPasswordException("Incorrect password");
        }
        catch (IOException e) {
            if (e.getCause() instanceof UnrecoverableKeyException) {
                throw new IncorrectPasswordException("Incorrect password");
            }
            log.error("Could not load key " + keyEntry.toString(), (Throwable)e);
            throw new RuntimeException("Could not load key " + keyEntry.toString(), e);
        }
        catch (Exception e) {
            log.error("Could not load key " + keyEntry.toString(), (Throwable)e);
            throw new RuntimeException("Could not load key " + keyEntry.toString(), e);
        }
    }

    public void saveKeyRing(KeyRing keyRing, String oldPassword, String password) {
        SecretKey symmetric = keyRing.getSymmetricKey();
        this.saveKey(symmetric, KeyEntry.SYM_ENCRYPTION.getAlias(), KeyEntry.SYM_ENCRYPTION.getFileName(), oldPassword, password);
        this.saveKey(keyRing.getSignatureKeyPair().getPrivate(), KeyEntry.MSG_SIGNATURE.getFileName(), symmetric);
        this.saveKey(keyRing.getEncryptionKeyPair().getPrivate(), KeyEntry.MSG_ENCRYPTION.getFileName(), symmetric);
    }

    private void saveKey(PrivateKey key, String fileName, SecretKey secretKey) {
        if (!this.storageDir.exists()) {
            this.storageDir.mkdirs();
        }
        PKCS8EncodedKeySpec pkcs8EncodedKeySpec = new PKCS8EncodedKeySpec(key.getEncoded());
        byte[] keyBytes = pkcs8EncodedKeySpec.getEncoded();
        try (FileOutputStream fos = new FileOutputStream(String.valueOf(this.storageDir) + "/" + fileName);){
            keyBytes = Encryption.encryptPayloadWithHmac(keyBytes, secretKey);
            fos.write(keyBytes);
        }
        catch (Exception e) {
            log.error("Could not save key " + fileName, (Throwable)e);
            throw new RuntimeException("Could not save key " + fileName, e);
        }
    }

    private void saveKey(SecretKey key, String alias, String fileName, String oldPassword, String password) {
        if (!this.storageDir.exists()) {
            this.storageDir.mkdirs();
        }
        if (password != null && !password.matches("\\p{ASCII}*")) {
            throw new IllegalArgumentException("Password must be ASCII.");
        }
        char[] oldPasswordChars = oldPassword == null ? new char[]{} : oldPassword.toCharArray();
        char[] passwordChars = password == null ? new char[]{} : password.toCharArray();
        try {
            String path = String.valueOf(this.storageDir) + "/" + fileName;
            KeyStore keyStore = KeyStore.getInstance("PKCS12");
            try {
                keyStore.load(new FileInputStream(path), oldPasswordChars);
            }
            catch (Exception e) {
                keyStore.load(null, null);
            }
            keyStore.setKeyEntry(alias, key, passwordChars, null);
            keyStore.store(new FileOutputStream(path), passwordChars);
        }
        catch (Exception e) {
            throw new RuntimeException("Could not save key " + alias, e);
        }
    }

    public static enum KeyEntry {
        SYM_ENCRYPTION("sym.p12", "AES", "sym"),
        MSG_SIGNATURE("sig.key", "DSA", "sig"),
        MSG_ENCRYPTION("enc.key", "RSA", "enc");

        private final String fileName;
        private final String algorithm;
        private final String alias;

        private KeyEntry(String fileName, String algorithm, String alias) {
            this.fileName = fileName;
            this.algorithm = algorithm;
            this.alias = alias;
        }

        public String getFileName() {
            return this.fileName;
        }

        public String getAlgorithm() {
            return this.algorithm;
        }

        public String getAlias() {
            return this.alias;
        }

        @NotNull
        public String toString() {
            return "Key{fileName='" + this.fileName + "', algorithm='" + this.algorithm + "'}";
        }
    }
}

