/*
 * Decompiled with CFR 0.152.
 */
package haveno.core.util;

import com.google.common.base.Preconditions;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import haveno.common.util.MathUtils;
import haveno.core.locale.CurrencyUtil;
import haveno.core.monetary.Price;
import haveno.core.offer.Offer;
import haveno.core.offer.OfferDirection;
import haveno.core.provider.price.MarketPrice;
import haveno.core.provider.price.PriceFeedService;
import haveno.core.trade.statistics.TradeStatisticsManager;
import haveno.core.user.Preferences;
import haveno.core.util.FormattingUtils;
import haveno.core.util.ParsingUtils;
import haveno.core.util.validation.AmountValidator4Decimals;
import haveno.core.util.validation.AmountValidator8Decimals;
import haveno.core.util.validation.InputValidator;
import haveno.core.util.validation.MonetaryValidator;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class PriceUtil {
    private static final Logger log = LoggerFactory.getLogger(PriceUtil.class);
    private final PriceFeedService priceFeedService;

    @Inject
    public PriceUtil(PriceFeedService priceFeedService, TradeStatisticsManager tradeStatisticsManager, Preferences preferences) {
        this.priceFeedService = priceFeedService;
    }

    public static MonetaryValidator getPriceValidator(String currencyCode) {
        return CurrencyUtil.isPricePrecise(currencyCode) ? new AmountValidator4Decimals() : new AmountValidator8Decimals();
    }

    public static InputValidator.ValidationResult isTriggerPriceValid(String triggerPriceAsString, MarketPrice marketPrice, boolean isSellOffer, String currencyCode) {
        if (triggerPriceAsString == null || triggerPriceAsString.isEmpty()) {
            return new InputValidator.ValidationResult(true);
        }
        InputValidator.ValidationResult result = PriceUtil.getPriceValidator(currencyCode).validate(triggerPriceAsString);
        if (!result.isValid) {
            return result;
        }
        return new InputValidator.ValidationResult(true);
    }

    public static Price marketPriceToPrice(MarketPrice marketPrice) {
        String currencyCode = marketPrice.getCurrencyCode();
        double priceAsDouble = marketPrice.getPrice();
        int precision = CurrencyUtil.isTraditionalCurrency(currencyCode) ? 8 : 8;
        double scaled = MathUtils.scaleUpByPowerOf10((double)priceAsDouble, (int)precision);
        long roundedToLong = MathUtils.roundDoubleToLong((double)scaled);
        return Price.valueOf(currencyCode, roundedToLong);
    }

    public boolean hasMarketPrice(Offer offer) {
        String currencyCode = offer.getCounterCurrencyCode();
        Preconditions.checkNotNull((Object)this.priceFeedService, (Object)"priceFeed must not be null");
        MarketPrice marketPrice = this.priceFeedService.getMarketPrice(currencyCode);
        Price price = offer.getPrice();
        return price != null && marketPrice != null && marketPrice.isRecentExternalPriceAvailable();
    }

    public Optional<Double> getMarketBasedPrice(Offer offer, OfferDirection direction) {
        if (offer.isUseMarketBasedPrice()) {
            return Optional.of(offer.getMarketPriceMarginPct());
        }
        if (!this.hasMarketPrice(offer)) {
            log.trace("We don't have a market price. That case could only happen if you don't have a price feed.");
            return Optional.empty();
        }
        String currencyCode = offer.getCounterCurrencyCode();
        Preconditions.checkNotNull((Object)this.priceFeedService, (Object)"priceFeed must not be null");
        MarketPrice marketPrice = this.priceFeedService.getMarketPrice(currencyCode);
        double marketPriceAsDouble = ((MarketPrice)Preconditions.checkNotNull((Object)marketPrice)).getPrice();
        return PriceUtil.calculatePercentage(offer, marketPriceAsDouble, direction);
    }

    public static Optional<Double> calculatePercentage(Offer offer, double marketPrice, OfferDirection direction) {
        double value;
        String currencyCode = offer.getCounterCurrencyCode();
        Price price = offer.getPrice();
        int precision = CurrencyUtil.isTraditionalCurrency(currencyCode) ? 8 : 8;
        long priceAsLong = ((Price)Preconditions.checkNotNull((Object)price)).getValue();
        double scaled = MathUtils.scaleDownByPowerOf10((long)priceAsLong, (int)precision);
        if (direction == OfferDirection.SELL) {
            if (marketPrice == 0.0) {
                return Optional.empty();
            }
            value = 1.0 - scaled / marketPrice;
        } else {
            if (marketPrice == 1.0) {
                return Optional.empty();
            }
            value = scaled / marketPrice - 1.0;
        }
        return Optional.of(value);
    }

    public static long getMarketPriceAsLong(String inputValue, String currencyCode) {
        if (inputValue == null || inputValue.isEmpty() || currencyCode == null) {
            return 0L;
        }
        try {
            int precision = PriceUtil.getMarketPricePrecision(currencyCode);
            String stringValue = PriceUtil.reformatMarketPrice(inputValue, currencyCode);
            return ParsingUtils.parsePriceStringToLong(currencyCode, stringValue, precision);
        }
        catch (Throwable t) {
            return 0L;
        }
    }

    public static String reformatMarketPrice(String inputValue, String currencyCode) {
        if (inputValue == null || inputValue.isEmpty() || currencyCode == null) {
            return "";
        }
        double priceAsDouble = ParsingUtils.parseNumberStringToDouble(inputValue);
        int precision = PriceUtil.getMarketPricePrecision(currencyCode);
        return FormattingUtils.formatRoundedDoubleWithPrecision(priceAsDouble, precision);
    }

    public static String formatMarketPrice(long priceAsLong, String currencyCode) {
        Price price = Price.valueOf(currencyCode, priceAsLong);
        return FormattingUtils.formatPrice(price);
    }

    public static int getMarketPricePrecision(String currencyCode) {
        return CurrencyUtil.isTraditionalCurrency(currencyCode) ? 8 : 8;
    }

    public static long invertLongPrice(long price, String currencyCode) {
        if (price == 0L) {
            return 0L;
        }
        int precision = CurrencyUtil.isTraditionalCurrency(currencyCode) ? 8 : 8;
        double priceDouble = MathUtils.scaleDownByPowerOf10((long)price, (int)precision);
        double priceDoubleInverted = BigDecimal.ONE.divide(BigDecimal.valueOf(priceDouble), precision, RoundingMode.HALF_UP).doubleValue();
        double scaled = MathUtils.scaleUpByPowerOf10((double)priceDoubleInverted, (int)precision);
        return MathUtils.roundDoubleToLong((double)scaled);
    }
}

