/*
 * Decompiled with CFR 0.152.
 */
package haveno.core.provider.price;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.SettableFuture;
import com.google.inject.Inject;
import haveno.common.Timer;
import haveno.common.UserThread;
import haveno.common.handlers.FaultHandler;
import haveno.common.util.MathUtils;
import haveno.core.locale.CurrencyUtil;
import haveno.core.locale.TradeCurrency;
import haveno.core.monetary.Price;
import haveno.core.provider.PriceHttpClient;
import haveno.core.provider.ProvidersRepository;
import haveno.core.provider.price.MarketPrice;
import haveno.core.provider.price.PriceProvider;
import haveno.core.provider.price.PriceRequest;
import haveno.core.provider.price.PriceRequestException;
import haveno.core.trade.statistics.TradeStatistics3;
import haveno.core.user.Preferences;
import haveno.network.http.HttpClient;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ReadOnlyIntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.beans.property.SimpleStringProperty;
import javafx.beans.property.StringProperty;
import javafx.beans.value.ChangeListener;
import javax.annotation.Nullable;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PriceFeedService {
    private static final Logger log = LoggerFactory.getLogger(PriceFeedService.class);
    private final HttpClient httpClient;
    private final ProvidersRepository providersRepository;
    private final Preferences preferences;
    private static final long PERIOD_SEC = 60L;
    private final Map<String, MarketPrice> cache = new HashMap<String, MarketPrice>();
    private PriceProvider priceProvider;
    @Nullable
    private Consumer<Double> priceConsumer;
    @Nullable
    private FaultHandler faultHandler;
    private String currencyCode;
    private final StringProperty currencyCodeProperty = new SimpleStringProperty();
    private final IntegerProperty updateCounter = new SimpleIntegerProperty(0);
    private long epochInMillisAtLastRequest;
    private long retryDelay = 0L;
    private long requestTs;
    private long lastLoopTs = System.currentTimeMillis();
    @Nullable
    private String baseUrlOfRespondingProvider;
    @Nullable
    private Timer requestTimer;
    @Nullable
    private PriceRequest priceRequest;
    private String requestAllPricesError = null;

    @Inject
    public PriceFeedService(PriceHttpClient httpClient, ProvidersRepository providersRepository, Preferences preferences) {
        this.httpClient = httpClient;
        this.providersRepository = providersRepository;
        this.preferences = preferences;
        this.priceProvider = new PriceProvider((HttpClient)httpClient, providersRepository.getBaseUrl());
    }

    public void shutDown() {
        log.info("Shutting down {}", (Object)this.getClass().getSimpleName());
        if (this.requestTimer != null) {
            this.requestTimer.stop();
            this.requestTimer = null;
        }
        if (this.priceRequest != null) {
            this.priceRequest.shutDown();
        }
    }

    public void setCurrencyCodeOnInit() {
        if (this.getCurrencyCode() == null) {
            TradeCurrency preferredTradeCurrency = this.preferences.getPreferredTradeCurrency();
            String code = preferredTradeCurrency != null ? preferredTradeCurrency.getCode() : "USD";
            this.setCurrencyCode(code);
        }
    }

    public void requestPrices() {
        this.request(false);
    }

    public void awaitExternalPrices() {
        CountDownLatch latch = new CountDownLatch(1);
        ChangeListener listener = (observable, oldValue, newValue) -> {
            if (this.hasExternalPrices()) {
                UserThread.execute(() -> latch.countDown());
            }
        };
        UserThread.execute(() -> this.updateCounter.addListener(listener));
        if (this.hasExternalPrices()) {
            UserThread.execute(() -> latch.countDown());
        }
        try {
            latch.await();
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        finally {
            UserThread.execute(() -> this.updateCounter.removeListener(listener));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean hasExternalPrices() {
        Map<String, MarketPrice> map = this.cache;
        synchronized (map) {
            return this.cache.values().stream().anyMatch(MarketPrice::isExternallyProvidedPrice);
        }
    }

    public void startRequestingPrices() {
        if (this.requestTimer == null) {
            this.request(true);
        }
    }

    public void startRequestingPrices(Consumer<Double> resultHandler, FaultHandler faultHandler) {
        this.priceConsumer = resultHandler;
        this.faultHandler = faultHandler;
        this.startRequestingPrices();
    }

    public String getProviderNodeAddress() {
        return this.httpClient.getBaseUrl();
    }

    private void request(boolean repeatRequests) {
        if (this.requestTs == 0L) {
            log.debug("request from provider {}", (Object)this.providersRepository.getBaseUrl());
        } else {
            log.debug("request from provider {} {} sec. after last request", (Object)this.providersRepository.getBaseUrl(), (Object)((double)(System.currentTimeMillis() - this.requestTs) / 1000.0));
        }
        this.requestTs = System.currentTimeMillis();
        this.baseUrlOfRespondingProvider = null;
        this.requestAllPrices(this.priceProvider, () -> {
            this.baseUrlOfRespondingProvider = this.priceProvider.getBaseUrl();
            boolean success = this.applyPriceToConsumer();
            if (success) {
                MarketPrice marketPrice = this.cache.get(this.currencyCode);
                if (marketPrice != null) {
                    log.debug("Received new {} from provider {} after {} sec.", new Object[]{marketPrice, this.baseUrlOfRespondingProvider, (double)(System.currentTimeMillis() - this.requestTs) / 1000.0});
                } else {
                    log.debug("Received new data from provider {} after {} sec. Requested market price for currency {} was not provided. That is expected if currency is not listed at provider.", new Object[]{this.baseUrlOfRespondingProvider, (double)(System.currentTimeMillis() - this.requestTs) / 1000.0, this.currencyCode});
                }
            } else {
                log.warn("applyPriceToConsumer was not successful. We retry with a new provider.");
                this.retryWithNewProvider();
            }
        }, (errorMessage, throwable) -> {
            if (throwable instanceof PriceRequestException) {
                String baseUrlOfFaultyRequest = ((PriceRequestException)throwable).priceProviderBaseUrl;
                String baseUrlOfCurrentRequest = this.priceProvider.getBaseUrl();
                if (baseUrlOfCurrentRequest.equals(baseUrlOfFaultyRequest)) {
                    log.info("We received an error requesting prices: baseUrlOfFaultyRequest={}, error={}", (Object)baseUrlOfFaultyRequest, (Object)throwable.toString());
                    this.retryWithNewProvider();
                } else {
                    log.debug("We received an error from an earlier request. We have started a new request already so we ignore that error. baseUrlOfCurrentRequest={}, baseUrlOfFaultyRequest={}", (Object)baseUrlOfCurrentRequest, (Object)baseUrlOfFaultyRequest);
                }
            } else {
                log.warn("We received an error with throwable={}", (Object)throwable.toString());
                this.retryWithNewProvider();
            }
            if (this.faultHandler != null) {
                this.faultHandler.handleFault(errorMessage, throwable);
            }
        });
        if (repeatRequests) {
            if (this.requestTimer != null) {
                this.requestTimer.stop();
            }
            long delay = 60L + (long)new Random().nextInt(5);
            this.requestTimer = UserThread.runAfter(() -> {
                if (this.baseUrlOfRespondingProvider == null) {
                    String oldBaseUrl = this.priceProvider.getBaseUrl();
                    this.setNewPriceProvider();
                    log.warn("We did not receive a response from provider {}. We select the new provider {} and use that for a new request.", (Object)oldBaseUrl, (Object)this.priceProvider.getBaseUrl());
                }
                this.request(true);
            }, (long)delay);
        }
    }

    private void retryWithNewProvider() {
        long thisRetryDelay = 0L;
        String oldBaseUrl = this.priceProvider.getBaseUrl();
        boolean looped = this.setNewPriceProvider();
        if (looped) {
            log.warn("Exhausted price provider list, looping to beginning");
            this.retryDelay = System.currentTimeMillis() - this.lastLoopTs < 60000L ? Math.min(this.retryDelay + 5L, 60L) : 0L;
            this.lastLoopTs = System.currentTimeMillis();
            thisRetryDelay = this.retryDelay;
        }
        log.info("We received an error at the request from provider {}. We select the new provider {} and use that for a new request in {} sec.", new Object[]{oldBaseUrl, this.priceProvider.getBaseUrl(), thisRetryDelay});
        if (thisRetryDelay > 0L) {
            UserThread.runAfter(() -> this.request(true), (long)thisRetryDelay);
        } else {
            this.request(true);
        }
    }

    private boolean setNewPriceProvider() {
        this.httpClient.cancelPendingRequest();
        boolean looped = this.providersRepository.selectNextProviderBaseUrl();
        if (!this.providersRepository.getBaseUrl().isEmpty()) {
            this.priceProvider = new PriceProvider(this.httpClient, this.providersRepository.getBaseUrl());
        } else {
            log.warn("We cannot create a new priceProvider because new base url is empty.");
        }
        return looped;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    public MarketPrice getMarketPrice(String currencyCode) {
        Map<String, MarketPrice> map = this.cache;
        synchronized (map) {
            return this.cache.getOrDefault(CurrencyUtil.getCurrencyCodeBase(currencyCode), null);
        }
    }

    private void setHavenoMarketPrice(String counterCurrencyCode, Price price) {
        UserThread.execute(() -> {
            String counterCurrencyCodeBase = CurrencyUtil.getCurrencyCodeBase(counterCurrencyCode);
            Map<String, MarketPrice> map = this.cache;
            synchronized (map) {
                if (!this.cache.containsKey(counterCurrencyCodeBase) || !this.cache.get(counterCurrencyCodeBase).isExternallyProvidedPrice()) {
                    this.cache.put(counterCurrencyCodeBase, new MarketPrice(counterCurrencyCodeBase, MathUtils.scaleDownByPowerOf10((long)price.getValue(), (int)(CurrencyUtil.isCryptoCurrency(counterCurrencyCode) ? 8 : 8)), 0L, false));
                }
                this.updateCounter.set(this.updateCounter.get() + 1);
            }
        });
    }

    public void setCurrencyCode(String currencyCode) {
        UserThread.await(() -> {
            if (this.currencyCode == null || !this.currencyCode.equals(currencyCode)) {
                this.currencyCode = currencyCode;
                this.currencyCodeProperty.set((Object)currencyCode);
                if (this.priceConsumer != null) {
                    this.applyPriceToConsumer();
                }
            }
        });
    }

    public String getCurrencyCode() {
        return this.currencyCode;
    }

    public StringProperty currencyCodeProperty() {
        return this.currencyCodeProperty;
    }

    public ReadOnlyIntegerProperty updateCounterProperty() {
        return this.updateCounter;
    }

    public Date getLastRequestTimeStamp() {
        return new Date(this.epochInMillisAtLastRequest);
    }

    public void applyLatestHavenoMarketPrice(Set<TradeStatistics3> tradeStatisticsSet) {
        HashMap mapByCurrencyCode = new HashMap();
        tradeStatisticsSet.forEach(e -> {
            List<TradeStatistics3> list;
            String currencyCode = e.getCurrency();
            if (mapByCurrencyCode.containsKey(currencyCode)) {
                list = (List)mapByCurrencyCode.get(currencyCode);
            } else {
                list = new ArrayList();
                mapByCurrencyCode.put(currencyCode, list);
            }
            list.add((TradeStatistics3)e);
        });
        mapByCurrencyCode.values().stream().filter(list -> !list.isEmpty()).forEach(list -> {
            list.sort(Comparator.comparing(TradeStatistics3::getDate));
            TradeStatistics3 tradeStatistics = (TradeStatistics3)list.get(list.size() - 1);
            this.setHavenoMarketPrice(tradeStatistics.getCurrency(), tradeStatistics.getTradePrice());
        });
    }

    public synchronized Map<String, MarketPrice> requestAllPrices() throws ExecutionException, InterruptedException, TimeoutException, CancellationException {
        CountDownLatch latch = new CountDownLatch(1);
        ChangeListener listener = (observable, oldValue, newValue) -> latch.countDown();
        UserThread.execute(() -> this.updateCounter.addListener(listener));
        this.requestAllPricesError = null;
        this.requestPrices();
        UserThread.runAfter(() -> {
            if (latch.getCount() > 0L) {
                this.requestAllPricesError = "Timeout fetching market prices within 20 seconds";
            }
            UserThread.execute(() -> latch.countDown());
        }, (long)20L);
        try {
            latch.await();
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        finally {
            UserThread.execute(() -> this.updateCounter.removeListener(listener));
        }
        if (this.requestAllPricesError != null) {
            throw new RuntimeException(this.requestAllPricesError);
        }
        return this.cache;
    }

    private boolean applyPriceToConsumer() {
        Object errorMessage;
        boolean result;
        block14: {
            result = false;
            errorMessage = null;
            if (this.currencyCode != null) {
                String baseUrl = this.priceProvider.getBaseUrl();
                this.httpClient.setBaseUrl(baseUrl);
                if (this.cache.containsKey(this.currencyCode)) {
                    try {
                        MarketPrice marketPrice = this.cache.get(this.currencyCode);
                        if (marketPrice.isExternallyProvidedPrice()) {
                            if (marketPrice.isRecentPriceAvailable()) {
                                if (this.priceConsumer != null) {
                                    this.priceConsumer.accept(marketPrice.getPrice());
                                }
                                result = true;
                            } else {
                                errorMessage = "Price for currency " + this.currencyCode + " is outdated by " + (Instant.now().getEpochSecond() - marketPrice.getTimestampSec()) / 60L + " minutes. Max. allowed age of price is 30 minutes. priceProvider=" + baseUrl + ". marketPrice= " + String.valueOf(marketPrice);
                            }
                            break block14;
                        }
                        if (this.baseUrlOfRespondingProvider == null) {
                            log.debug("Market price for currency " + this.currencyCode + " was not delivered by provider " + baseUrl + ". That is expected at startup.");
                        } else {
                            log.debug("Market price for currency " + this.currencyCode + " is not provided by the provider " + baseUrl + ". That is expected for currencies not listed at providers.");
                        }
                        result = true;
                    }
                    catch (Throwable t) {
                        errorMessage = "Exception at applyPriceToConsumer for currency " + this.currencyCode + ". priceProvider=" + baseUrl + ". Exception=" + String.valueOf(t);
                    }
                } else {
                    log.debug("We don't have a price for currency " + this.currencyCode + ". priceProvider=" + baseUrl + ". That is expected for currencies not listed at providers.");
                    result = true;
                }
            } else {
                errorMessage = "We don't have a currency yet set. That should never happen";
            }
        }
        if (errorMessage != null) {
            log.warn((String)errorMessage);
            if (this.faultHandler != null) {
                this.faultHandler.handleFault((String)errorMessage, (Throwable)new PriceRequestException((String)errorMessage));
            }
        }
        UserThread.execute(() -> this.updateCounter.set(this.updateCounter.get() + 1));
        return result;
    }

    private void requestAllPrices(PriceProvider provider, final Runnable resultHandler, final FaultHandler faultHandler) {
        if (this.httpClient.hasPendingRequest()) {
            log.warn("We have a pending request open. We ignore that request. httpClient {}", (Object)this.httpClient);
            return;
        }
        this.priceRequest = new PriceRequest();
        SettableFuture<Map<String, MarketPrice>> future = this.priceRequest.requestAllPrices(provider);
        Futures.addCallback(future, (FutureCallback)new FutureCallback<Map<String, MarketPrice>>(){

            public void onSuccess(@Nullable Map<String, MarketPrice> result) {
                UserThread.execute(() -> {
                    Preconditions.checkNotNull((Object)result, (Object)"Result must not be null at requestAllPrices");
                    PriceFeedService.this.epochInMillisAtLastRequest = System.currentTimeMillis();
                    Map priceMap = result;
                    Map<String, MarketPrice> map = PriceFeedService.this.cache;
                    synchronized (map) {
                        PriceFeedService.this.cache.putAll(priceMap);
                    }
                    resultHandler.run();
                });
            }

            public void onFailure(@NotNull Throwable throwable) {
                UserThread.execute(() -> faultHandler.handleFault("Could not load marketPrices", throwable));
            }
        }, (Executor)MoreExecutors.directExecutor());
    }
}

