/*
 * Decompiled with CFR 0.152.
 */
package haveno.core.account.witness;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.inject.Inject;
import haveno.common.UserThread;
import haveno.common.crypto.CryptoException;
import haveno.common.crypto.Hash;
import haveno.common.crypto.KeyRing;
import haveno.common.crypto.PubKeyRing;
import haveno.common.crypto.Sig;
import haveno.common.handlers.ErrorMessageHandler;
import haveno.common.util.MathUtils;
import haveno.common.util.Tuple2;
import haveno.common.util.Utilities;
import haveno.core.account.sign.SignedWitness;
import haveno.core.account.sign.SignedWitnessService;
import haveno.core.account.witness.AccountAgeWitness;
import haveno.core.account.witness.AccountAgeWitnessStorageService;
import haveno.core.account.witness.AccountAgeWitnessUtils;
import haveno.core.filter.FilterManager;
import haveno.core.locale.CurrencyUtil;
import haveno.core.locale.Res;
import haveno.core.offer.Offer;
import haveno.core.offer.OfferDirection;
import haveno.core.offer.OfferRestrictions;
import haveno.core.payment.ChargeBackRisk;
import haveno.core.payment.PaymentAccount;
import haveno.core.payment.TradeLimits;
import haveno.core.payment.payload.PaymentAccountPayload;
import haveno.core.payment.payload.PaymentMethod;
import haveno.core.support.dispute.Dispute;
import haveno.core.support.dispute.DisputeResult;
import haveno.core.support.dispute.arbitration.TraderDataItem;
import haveno.core.trade.ArbitratorTrade;
import haveno.core.trade.Trade;
import haveno.core.trade.protocol.TradePeer;
import haveno.core.user.User;
import haveno.network.p2p.BootstrapListener;
import haveno.network.p2p.P2PService;
import haveno.network.p2p.P2PServiceListener;
import haveno.network.p2p.storage.P2PDataStorage;
import haveno.network.p2p.storage.payload.PersistableNetworkPayload;
import haveno.network.p2p.storage.persistence.AppendOnlyDataStoreService;
import haveno.network.p2p.storage.persistence.MapStoreService;
import java.math.BigInteger;
import java.security.PublicKey;
import java.time.Clock;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.bitcoinj.core.ECKey;
import org.bitcoinj.core.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AccountAgeWitnessService {
    private static final Logger log = LoggerFactory.getLogger(AccountAgeWitnessService.class);
    private static final Date RELEASE = Utilities.getUTCDate((int)2017, (int)10, (int)11);
    private static final long SAFE_ACCOUNT_AGE_DATE = Utilities.getUTCDate((int)2019, (int)2, (int)1).getTime();
    private final KeyRing keyRing;
    private final P2PService p2PService;
    private final User user;
    private final SignedWitnessService signedWitnessService;
    private final ChargeBackRisk chargeBackRisk;
    private final AccountAgeWitnessStorageService accountAgeWitnessStorageService;
    private final Clock clock;
    private final FilterManager filterManager;
    private final AccountAgeWitnessUtils accountAgeWitnessUtils;
    private final Map<P2PDataStorage.ByteArray, AccountAgeWitness> accountAgeWitnessMap = new HashMap<P2PDataStorage.ByteArray, AccountAgeWitness>();
    private final Map<P2PDataStorage.ByteArray, AccountAgeWitness> accountAgeWitnessCache = new ConcurrentHashMap<P2PDataStorage.ByteArray, AccountAgeWitness>();

    @Inject
    public AccountAgeWitnessService(KeyRing keyRing, P2PService p2PService, User user, SignedWitnessService signedWitnessService, ChargeBackRisk chargeBackRisk, AccountAgeWitnessStorageService accountAgeWitnessStorageService, AppendOnlyDataStoreService appendOnlyDataStoreService, Clock clock, FilterManager filterManager) {
        this.keyRing = keyRing;
        this.p2PService = p2PService;
        this.user = user;
        this.signedWitnessService = signedWitnessService;
        this.chargeBackRisk = chargeBackRisk;
        this.accountAgeWitnessStorageService = accountAgeWitnessStorageService;
        this.clock = clock;
        this.filterManager = filterManager;
        this.accountAgeWitnessUtils = new AccountAgeWitnessUtils(this, signedWitnessService, keyRing);
        appendOnlyDataStoreService.addService((MapStoreService)accountAgeWitnessStorageService);
    }

    public void onAllServicesInitialized() {
        this.p2PService.getP2PDataStorage().addAppendOnlyDataStoreListener(payload -> {
            if (payload instanceof AccountAgeWitness) {
                this.addToMap((AccountAgeWitness)payload);
            }
        });
        this.accountAgeWitnessStorageService.getMapOfAllData().values().stream().filter(e -> e instanceof AccountAgeWitness).map(e -> (AccountAgeWitness)e).forEach(this::addToMap);
        if (this.p2PService.isBootstrapped()) {
            this.onBootStrapped();
        } else {
            this.p2PService.addP2PServiceListener((P2PServiceListener)new BootstrapListener(){

                public void onDataReceived() {
                    AccountAgeWitnessService.this.onBootStrapped();
                }
            });
        }
    }

    private void onBootStrapped() {
        this.republishAllTraditionalAccounts();
        this.signAndPublishSameNameAccounts();
    }

    private void republishAllTraditionalAccounts() {
        if (this.user.getPaymentAccounts() != null) {
            this.user.getPaymentAccounts().stream().filter(account -> account.getPaymentMethod().isTraditional()).forEach(account -> {
                AccountAgeWitness myWitness = this.getMyWitness(account.getPaymentAccountPayload());
                if (myWitness.isDateInTolerance(this.clock)) {
                    int delayInSec = 20 + new Random().nextInt(40);
                    UserThread.runAfter(() -> this.p2PService.addPersistableNetworkPayload((PersistableNetworkPayload)myWitness, true), (long)delayInSec);
                }
            });
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    public void addToMap(AccountAgeWitness accountAgeWitness) {
        AccountAgeWitnessService accountAgeWitnessService = this;
        synchronized (accountAgeWitnessService) {
            this.accountAgeWitnessMap.putIfAbsent(accountAgeWitness.getHashAsByteArray(), accountAgeWitness);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void publishMyAccountAgeWitness(PaymentAccountPayload paymentAccountPayload) {
        AccountAgeWitnessService accountAgeWitnessService = this;
        synchronized (accountAgeWitnessService) {
            AccountAgeWitness accountAgeWitness = this.getMyWitness(paymentAccountPayload);
            P2PDataStorage.ByteArray hash = accountAgeWitness.getHashAsByteArray();
            if (this.accountAgeWitnessCache.containsKey(hash)) {
                return;
            }
            if (!this.accountAgeWitnessMap.containsKey(hash)) {
                this.p2PService.addPersistableNetworkPayload((PersistableNetworkPayload)accountAgeWitness, false);
            }
        }
    }

    public byte[] getPeerAccountAgeWitnessHash(Trade trade) {
        return this.findTradePeerWitness(trade).map(AccountAgeWitness::getHash).orElse(null);
    }

    byte[] getAccountInputDataWithSalt(PaymentAccountPayload paymentAccountPayload) {
        return Utilities.concatenateByteArrays((byte[])paymentAccountPayload.getAgeWitnessInputData(), (byte[])paymentAccountPayload.getSalt());
    }

    @VisibleForTesting
    public AccountAgeWitness getNewWitness(PaymentAccountPayload paymentAccountPayload, PubKeyRing pubKeyRing) {
        byte[] accountInputDataWithSalt = this.getAccountInputDataWithSalt(paymentAccountPayload);
        byte[] hash = Hash.getSha256Ripemd160hash((byte[])Utilities.concatenateByteArrays((byte[])accountInputDataWithSalt, (byte[])pubKeyRing.getSignaturePubKeyBytes()));
        return new AccountAgeWitness(hash, new Date().getTime());
    }

    public Optional<AccountAgeWitness> findWitness(PaymentAccountPayload paymentAccountPayload, PubKeyRing pubKeyRing) {
        if (paymentAccountPayload == null) {
            return Optional.empty();
        }
        byte[] accountInputDataWithSalt = this.getAccountInputDataWithSalt(paymentAccountPayload);
        byte[] hash = Hash.getSha256Ripemd160hash((byte[])Utilities.concatenateByteArrays((byte[])accountInputDataWithSalt, (byte[])pubKeyRing.getSignaturePubKeyBytes()));
        return this.getWitnessByHash(hash);
    }

    public Optional<AccountAgeWitness> findWitness(Offer offer) {
        Optional<String> accountAgeWitnessHash = offer.getAccountAgeWitnessHashAsHex();
        return accountAgeWitnessHash.isPresent() ? this.getWitnessByHashAsHex(accountAgeWitnessHash.get()) : Optional.empty();
    }

    private Optional<AccountAgeWitness> findTradePeerWitness(Trade trade) {
        if (trade instanceof ArbitratorTrade) {
            return Optional.empty();
        }
        TradePeer tradePeer = trade.getTradePeer();
        return tradePeer == null || tradePeer.getPaymentAccountPayload() == null || tradePeer.getPubKeyRing() == null ? Optional.empty() : this.findWitness(tradePeer.getPaymentAccountPayload(), tradePeer.getPubKeyRing());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Optional<AccountAgeWitness> getWitnessByHash(byte[] hash) {
        P2PDataStorage.ByteArray hashAsByteArray = new P2PDataStorage.ByteArray(hash);
        AccountAgeWitnessService accountAgeWitnessService = this;
        synchronized (accountAgeWitnessService) {
            if (this.accountAgeWitnessCache.containsKey(hashAsByteArray)) {
                return Optional.of(this.accountAgeWitnessCache.get(hashAsByteArray));
            }
            if (this.accountAgeWitnessMap.containsKey(hashAsByteArray)) {
                AccountAgeWitness accountAgeWitness = this.accountAgeWitnessMap.get(hashAsByteArray);
                this.accountAgeWitnessCache.put(hashAsByteArray, accountAgeWitness);
                return Optional.of(accountAgeWitness);
            }
            return Optional.empty();
        }
    }

    private Optional<AccountAgeWitness> getWitnessByHashAsHex(String hashAsHex) {
        return this.getWitnessByHash(Utilities.decodeFromHex((String)hashAsHex));
    }

    public long getAccountAge(AccountAgeWitness accountAgeWitness, Date now) {
        log.debug("getAccountAge now={}, accountAgeWitness.getDate()={}", (Object)now.getTime(), (Object)accountAgeWitness.getDate());
        return now.getTime() - accountAgeWitness.getDate();
    }

    public long getAccountAge(PaymentAccountPayload paymentAccountPayload, PubKeyRing pubKeyRing) {
        return this.findWitness(paymentAccountPayload, pubKeyRing).map(accountAgeWitness -> this.getAccountAge((AccountAgeWitness)accountAgeWitness, new Date())).orElse(-1L);
    }

    public long getAccountAge(Offer offer) {
        return this.findWitness(offer).map(accountAgeWitness -> this.getAccountAge((AccountAgeWitness)accountAgeWitness, new Date())).orElse(-1L);
    }

    public long getAccountAge(Trade trade) {
        return this.findTradePeerWitness(trade).map(accountAgeWitness -> this.getAccountAge((AccountAgeWitness)accountAgeWitness, new Date())).orElse(-1L);
    }

    public long getWitnessSignAge(AccountAgeWitness accountAgeWitness, Date now) {
        List<Long> dates = this.signedWitnessService.getVerifiedWitnessDateList(accountAgeWitness);
        if (dates.isEmpty()) {
            return -1L;
        }
        return now.getTime() - dates.get(0);
    }

    public long getWitnessSignAge(Offer offer, Date now) {
        return this.findWitness(offer).map(witness -> this.getWitnessSignAge((AccountAgeWitness)witness, now)).orElse(-1L);
    }

    public long getWitnessSignAge(Trade trade, Date now) {
        return this.findTradePeerWitness(trade).map(witness -> this.getWitnessSignAge((AccountAgeWitness)witness, now)).orElse(-1L);
    }

    public AccountAge getPeersAccountAgeCategory(long peersAccountAge) {
        return this.getAccountAgeCategory(peersAccountAge);
    }

    private AccountAge getAccountAgeCategory(long accountAge) {
        if (accountAge < 0L) {
            return AccountAge.UNVERIFIED;
        }
        if (accountAge < TimeUnit.DAYS.toMillis(30L)) {
            return AccountAge.LESS_ONE_MONTH;
        }
        if (accountAge < TimeUnit.DAYS.toMillis(60L)) {
            return AccountAge.ONE_TO_TWO_MONTHS;
        }
        return AccountAge.TWO_MONTHS_OR_MORE;
    }

    private BigInteger getTradeLimit(BigInteger maxTradeLimit, String currencyCode, AccountAgeWitness accountAgeWitness, AccountAge accountAgeCategory, OfferDirection direction, PaymentMethod paymentMethod) {
        if (CurrencyUtil.isCryptoCurrency(currencyCode) || !PaymentMethod.hasChargebackRisk(paymentMethod, currencyCode) || direction == OfferDirection.SELL) {
            return maxTradeLimit;
        }
        BigInteger limit = OfferRestrictions.TOLERATED_SMALL_TRADE_AMOUNT;
        double factor = this.signedBuyFactor(accountAgeCategory);
        if (factor > 0.0) {
            limit = BigInteger.valueOf(MathUtils.roundDoubleToLong((double)((double)maxTradeLimit.longValueExact() * factor)));
        }
        if (accountAgeWitness != null) {
            log.debug("limit={}, factor={}, accountAgeWitnessHash={}", new Object[]{limit, factor, Utilities.bytesAsHexString((byte[])accountAgeWitness.getHash())});
        }
        return limit;
    }

    private double signedBuyFactor(AccountAge accountAgeCategory) {
        switch (accountAgeCategory.ordinal()) {
            case 3: {
                return 1.0;
            }
            case 2: {
                return 0.5;
            }
        }
        return 0.0;
    }

    private double normalFactor() {
        return 1.0;
    }

    private boolean isImmature(AccountAgeWitness accountAgeWitness) {
        return accountAgeWitness.getDate() > SAFE_ACCOUNT_AGE_DATE;
    }

    public boolean myHasTradeLimitException(PaymentAccount myPaymentAccount) {
        return this.hasTradeLimitException(this.getMyWitness(myPaymentAccount.getPaymentAccountPayload()));
    }

    private boolean hasTradeLimitException(AccountAgeWitness accountAgeWitness) {
        return !this.isImmature(accountAgeWitness) || this.signedWitnessService.isSignedByArbitrator(accountAgeWitness);
    }

    public AccountAgeWitness getMyWitness(PaymentAccountPayload paymentAccountPayload) {
        Optional<AccountAgeWitness> accountAgeWitnessOptional = this.findWitness(paymentAccountPayload, this.keyRing.getPubKeyRing());
        return accountAgeWitnessOptional.orElseGet(() -> this.getNewWitness(paymentAccountPayload, this.keyRing.getPubKeyRing()));
    }

    private byte[] getMyWitnessHash(PaymentAccountPayload paymentAccountPayload) {
        return this.getMyWitness(paymentAccountPayload).getHash();
    }

    public String getMyWitnessHashAsHex(PaymentAccountPayload paymentAccountPayload) {
        return Utilities.bytesAsHexString((byte[])this.getMyWitnessHash(paymentAccountPayload));
    }

    public long getMyAccountAge(PaymentAccountPayload paymentAccountPayload) {
        return this.getAccountAge(this.getMyWitness(paymentAccountPayload), new Date());
    }

    public long getMyTradeLimit(PaymentAccount paymentAccount, String currencyCode, OfferDirection direction, boolean buyerAsTakerWithoutDeposit) {
        if (paymentAccount == null) {
            return 0L;
        }
        if (buyerAsTakerWithoutDeposit) {
            TradeLimits tradeLimits = new TradeLimits();
            return tradeLimits.getMaxTradeLimitBuyerAsTakerWithoutDeposit().longValueExact();
        }
        AccountAgeWitness accountAgeWitness = this.getMyWitness(paymentAccount.getPaymentAccountPayload());
        BigInteger maxTradeLimit = paymentAccount.getPaymentMethod().getMaxTradeLimit(currencyCode);
        if (this.hasTradeLimitException(accountAgeWitness)) {
            return maxTradeLimit.longValueExact();
        }
        long accountSignAge = this.getWitnessSignAge(accountAgeWitness, new Date());
        AccountAge accountAgeCategory = this.getAccountAgeCategory(accountSignAge);
        return this.getTradeLimit(maxTradeLimit, currencyCode, accountAgeWitness, accountAgeCategory, direction, paymentAccount.getPaymentMethod()).longValueExact();
    }

    public long getUnsignedTradeLimit(PaymentMethod paymentMethod, String currencyCode, OfferDirection direction) {
        return this.getTradeLimit(paymentMethod.getMaxTradeLimit(currencyCode), currencyCode, null, AccountAge.UNVERIFIED, direction, paymentMethod).longValueExact();
    }

    public boolean verifyAccountAgeWitness(Trade trade, PaymentAccountPayload peersPaymentAccountPayload, PubKeyRing peersPubKeyRing, byte[] nonce, byte[] signature, ErrorMessageHandler errorMessageHandler) {
        AccountAgeWitness peersWitness;
        log.info("Verifying account age witness for {} {}, payment account payload hash={}, nonce={}, signature={}", new Object[]{trade.getClass().getSimpleName(), trade.getId(), Utilities.bytesAsHexString((byte[])peersPaymentAccountPayload.getHash()), Utilities.bytesAsHexString((byte[])nonce), Utilities.bytesAsHexString((byte[])signature)});
        Optional<AccountAgeWitness> accountAgeWitnessOptional = this.findWitness(peersPaymentAccountPayload, peersPubKeyRing);
        if (accountAgeWitnessOptional.isPresent()) {
            peersWitness = accountAgeWitnessOptional.get();
        } else {
            log.warn("We did not find the peers witness data. That is expected with peers using an older version.");
            peersWitness = this.getNewWitness(peersPaymentAccountPayload, peersPubKeyRing);
        }
        if (!this.isDateAfterReleaseDate(peersWitness.getDate(), RELEASE, errorMessageHandler)) {
            return false;
        }
        byte[] peersAccountInputDataWithSalt = Utilities.concatenateByteArrays((byte[])peersPaymentAccountPayload.getAgeWitnessInputData(), (byte[])peersPaymentAccountPayload.getSalt());
        byte[] hash = Hash.getSha256Ripemd160hash((byte[])Utilities.concatenateByteArrays((byte[])peersAccountInputDataWithSalt, (byte[])peersPubKeyRing.getSignaturePubKeyBytes()));
        byte[] peersWitnessHash = peersWitness.getHash();
        if (!this.verifyWitnessHash(peersWitnessHash, hash, errorMessageHandler)) {
            return false;
        }
        if (!this.verifyPeersTradeLimit(trade.getOffer(), trade.getAmount(), peersWitness, new Date(), errorMessageHandler)) {
            log.error("verifyPeersTradeLimit failed: peersPaymentAccountPayload {}", (Object)peersPaymentAccountPayload);
            return false;
        }
        return this.verifySignature(peersPubKeyRing.getSignaturePubKey(), nonce, signature, errorMessageHandler);
    }

    public boolean verifyPeersTradeAmount(Offer offer, BigInteger tradeAmount, ErrorMessageHandler errorMessageHandler) {
        Preconditions.checkNotNull((Object)offer);
        return this.findWitness(offer).map(witness -> this.verifyPeersTradeLimit(offer, tradeAmount, (AccountAgeWitness)witness, new Date(), errorMessageHandler)).orElse(this.isToleratedSmalleAmount(tradeAmount));
    }

    private boolean isToleratedSmalleAmount(BigInteger tradeAmount) {
        return tradeAmount.longValueExact() <= OfferRestrictions.TOLERATED_SMALL_TRADE_AMOUNT.longValueExact();
    }

    boolean isDateAfterReleaseDate(long witnessDateAsLong, Date ageWitnessReleaseDate, ErrorMessageHandler errorMessageHandler) {
        Date witnessDate = new Date(witnessDateAsLong);
        Date releaseDateWithTolerance = new Date(ageWitnessReleaseDate.getTime() - TimeUnit.DAYS.toMillis(1L));
        boolean result = witnessDate.after(releaseDateWithTolerance);
        if (!result) {
            String msg = "Witness date is set earlier than release date of ageWitness feature. ageWitnessReleaseDate=" + String.valueOf(ageWitnessReleaseDate) + ", witnessDate=" + String.valueOf(witnessDate);
            log.warn(msg);
            errorMessageHandler.handleErrorMessage(msg);
        }
        return result;
    }

    public boolean verifyPeersCurrentDate(Date peersCurrentDate) {
        boolean result;
        boolean bl = result = Math.abs(peersCurrentDate.getTime() - new Date().getTime()) <= TimeUnit.DAYS.toMillis(1L);
        if (!result) {
            String msg = "Peers current date is further than 1 day off to our current date. PeersCurrentDate=" + String.valueOf(peersCurrentDate) + "; myCurrentDate=" + String.valueOf(new Date());
            throw new RuntimeException(msg);
        }
        return result;
    }

    private boolean verifyWitnessHash(byte[] witnessHash, byte[] hash, ErrorMessageHandler errorMessageHandler) {
        boolean result = Arrays.equals(witnessHash, hash);
        if (!result) {
            String msg = "witnessHash is not matching peers hash. witnessHash=" + Utilities.bytesAsHexString((byte[])witnessHash) + ", hash=" + Utilities.bytesAsHexString((byte[])hash);
            log.warn(msg);
            errorMessageHandler.handleErrorMessage(msg);
        }
        return result;
    }

    private boolean verifyPeersTradeLimit(Offer offer, BigInteger tradeAmount, AccountAgeWitness peersWitness, Date peersCurrentDate, ErrorMessageHandler errorMessageHandler) {
        boolean result;
        BigInteger defaultMaxTradeLimit;
        Preconditions.checkNotNull((Object)offer);
        String currencyCode = offer.getCounterCurrencyCode();
        BigInteger peersCurrentTradeLimit = defaultMaxTradeLimit = offer.getPaymentMethod().getMaxTradeLimit(currencyCode);
        if (!this.hasTradeLimitException(peersWitness)) {
            long accountSignAge = this.getWitnessSignAge(peersWitness, peersCurrentDate);
            AccountAge accountAgeCategory = this.getPeersAccountAgeCategory(accountSignAge);
            OfferDirection direction = offer.isMyOffer(this.keyRing) ? offer.getMirroredDirection() : offer.getDirection();
            peersCurrentTradeLimit = this.getTradeLimit(defaultMaxTradeLimit, currencyCode, peersWitness, accountAgeCategory, direction, offer.getPaymentMethod());
        }
        boolean bl = result = tradeAmount.longValueExact() <= peersCurrentTradeLimit.longValueExact();
        if (!result) {
            String msg = "The peers trade limit is less than the traded amount.\ntradeAmount=" + String.valueOf(tradeAmount) + "\nPeers trade limit=" + String.valueOf(peersCurrentTradeLimit) + "\nOffer ID=" + offer.getShortId() + "\nPaymentMethod=" + offer.getPaymentMethod().getId() + "\nCurrencyCode=" + offer.getCounterCurrencyCode();
            log.warn(msg);
            errorMessageHandler.handleErrorMessage(msg);
        }
        return result;
    }

    boolean verifySignature(PublicKey peersPublicKey, byte[] nonce, byte[] signature, ErrorMessageHandler errorMessageHandler) {
        boolean result;
        try {
            result = Sig.verify((PublicKey)peersPublicKey, (byte[])nonce, (byte[])signature);
        }
        catch (CryptoException e) {
            log.warn(e.toString());
            result = false;
        }
        if (!result) {
            String msg = "Signature of nonce is not correct. peersPublicKey=" + String.valueOf(peersPublicKey) + ", nonce(hex)=" + Utilities.bytesAsHexString((byte[])nonce) + ", signature=" + Utilities.bytesAsHexString((byte[])signature);
            log.warn(msg);
            errorMessageHandler.handleErrorMessage(msg);
        }
        return result;
    }

    public void arbitratorSignAccountAgeWitness(BigInteger tradeAmount, AccountAgeWitness accountAgeWitness, ECKey key, PublicKey peersPubKey) {
        this.signedWitnessService.signAndPublishAccountAgeWitness(tradeAmount, accountAgeWitness, key, peersPubKey);
    }

    public String arbitratorSignOrphanWitness(AccountAgeWitness accountAgeWitness, ECKey ecKey, long time) {
        return this.signedWitnessService.getSignedWitnessSet(accountAgeWitness).stream().findAny().map(SignedWitness::getWitnessOwnerPubKey).map(witnessOwnerPubKey -> this.signedWitnessService.signAndPublishAccountAgeWitness(accountAgeWitness, ecKey, (byte[])witnessOwnerPubKey, time)).orElse("No signedWitness found");
    }

    public String arbitratorSignOrphanPubKey(ECKey key, byte[] peersPubKey, long childSignTime) {
        return this.signedWitnessService.signTraderPubKey(key, peersPubKey, childSignTime);
    }

    public void arbitratorSignAccountAgeWitness(AccountAgeWitness accountAgeWitness, ECKey key, byte[] tradersPubKey, long time) {
        this.signedWitnessService.signAndPublishAccountAgeWitness(accountAgeWitness, key, tradersPubKey, time);
    }

    public Optional<SignedWitness> traderSignAndPublishPeersAccountAgeWitness(Trade trade) {
        Preconditions.checkNotNull((Object)trade.getTradePeer().getPubKeyRing(), (Object)"Peer must have a keyring");
        PublicKey peersPubKey = trade.getTradePeer().getPubKeyRing().getSignaturePubKey();
        Preconditions.checkNotNull((Object)peersPubKey, (Object)"Peers pub key must not be null");
        AccountAgeWitness peersWitness = this.findTradePeerWitness(trade).orElse(null);
        Preconditions.checkNotNull((Object)peersWitness, (Object)("Not able to find peers witness, unable to sign for trade " + trade.toString()));
        BigInteger tradeAmount = trade.getAmount();
        Preconditions.checkNotNull((Object)tradeAmount, (Object)"Trade amount must not be null");
        try {
            return this.signedWitnessService.signAndPublishAccountAgeWitness(tradeAmount, peersWitness, peersPubKey);
        }
        catch (CryptoException e) {
            log.warn("Trader failed to sign witness, exception {}", (Object)e.toString());
            return Optional.empty();
        }
    }

    public boolean publishOwnSignedWitness(SignedWitness signedWitness) {
        return this.signedWitnessService.publishOwnSignedWitness(signedWitness);
    }

    public List<TraderDataItem> getTraderPaymentAccounts(long safeDate, PaymentMethod paymentMethod, List<Dispute> disputes) {
        return disputes.stream().filter(dispute -> dispute.getContract().getPaymentMethodId().equals(paymentMethod.getId())).filter(this::isNotFiltered).filter(this::hasChargebackRisk).filter(this::isBuyerWinner).flatMap(this::getTraderData).filter(Objects::nonNull).filter(traderDataItem -> !this.signedWitnessService.isSignedAccountAgeWitness(traderDataItem.getAccountAgeWitness())).filter(traderDataItem -> traderDataItem.getAccountAgeWitness().getDate() < safeDate).distinct().collect(Collectors.toList());
    }

    private boolean isNotFiltered(Dispute dispute) {
        boolean isFiltered = this.filterManager.isNodeAddressBanned(dispute.getContract().getBuyerNodeAddress()) || this.filterManager.isNodeAddressBanned(dispute.getContract().getSellerNodeAddress()) || this.filterManager.isCurrencyBanned(dispute.getContract().getOfferPayload().getCurrencyCode()) || this.filterManager.isPaymentMethodBanned(PaymentMethod.getPaymentMethodOrNA(dispute.getContract().getPaymentMethodId())) || this.filterManager.arePeersPaymentAccountDataBanned(dispute.getBuyerPaymentAccountPayload()) || this.filterManager.arePeersPaymentAccountDataBanned(dispute.getSellerPaymentAccountPayload()) || this.filterManager.isWitnessSignerPubKeyBanned(Utils.HEX.encode(dispute.getContract().getBuyerPubKeyRing().getSignaturePubKeyBytes())) || this.filterManager.isWitnessSignerPubKeyBanned(Utils.HEX.encode(dispute.getContract().getSellerPubKeyRing().getSignaturePubKeyBytes()));
        return !isFiltered;
    }

    @VisibleForTesting
    public boolean hasChargebackRisk(Dispute dispute) {
        return this.chargeBackRisk.hasChargebackRisk(dispute.getContract().getPaymentMethodId(), dispute.getContract().getOfferPayload().getCurrencyCode());
    }

    private boolean isBuyerWinner(Dispute dispute) {
        if (!dispute.isClosed() || dispute.getDisputeResultProperty() == null) {
            return false;
        }
        return ((DisputeResult)dispute.getDisputeResultProperty().get()).getWinner() == DisputeResult.Winner.BUYER;
    }

    private Stream<TraderDataItem> getTraderData(Dispute dispute) {
        BigInteger tradeAmount = dispute.getContract().getTradeAmount();
        PubKeyRing buyerPubKeyRing = dispute.getContract().getBuyerPubKeyRing();
        PubKeyRing sellerPubKeyRing = dispute.getContract().getSellerPubKeyRing();
        PaymentAccountPayload buyerPaymentAccountPaload = dispute.getBuyerPaymentAccountPayload();
        PaymentAccountPayload sellerPaymentAccountPaload = dispute.getSellerPaymentAccountPayload();
        TraderDataItem buyerData = this.findWitness(buyerPaymentAccountPaload, buyerPubKeyRing).map(witness -> new TraderDataItem(buyerPaymentAccountPaload, (AccountAgeWitness)witness, tradeAmount, buyerPubKeyRing.getSignaturePubKey())).orElse(null);
        TraderDataItem sellerData = this.findWitness(sellerPaymentAccountPaload, sellerPubKeyRing).map(witness -> new TraderDataItem(sellerPaymentAccountPaload, (AccountAgeWitness)witness, tradeAmount, sellerPubKeyRing.getSignaturePubKey())).orElse(null);
        return Stream.of(buyerData, sellerData);
    }

    public boolean hasSignedWitness(Offer offer) {
        return this.findWitness(offer).map(this.signedWitnessService::isSignedAccountAgeWitness).orElse(false);
    }

    public boolean peerHasSignedWitness(Trade trade) {
        return this.findTradePeerWitness(trade).map(this.signedWitnessService::isSignedAccountAgeWitness).orElse(false);
    }

    public boolean accountIsSigner(AccountAgeWitness accountAgeWitness) {
        return this.signedWitnessService.isSignerAccountAgeWitness(accountAgeWitness);
    }

    public boolean tradeAmountIsSufficient(BigInteger tradeAmount) {
        return this.signedWitnessService.isSufficientTradeAmountForSigning(tradeAmount);
    }

    public SignState getSignState(Offer offer) {
        return this.findWitness(offer).map(this::getSignState).orElse(SignState.UNSIGNED);
    }

    public SignState getSignState(Trade trade) {
        if (trade instanceof ArbitratorTrade) {
            return SignState.UNSIGNED;
        }
        return this.findTradePeerWitness(trade).map(this::getSignState).orElse(SignState.UNSIGNED);
    }

    public SignState getSignState(AccountAgeWitness accountAgeWitness) {
        String hash;
        String string = hash = log.isDebugEnabled() ? Utilities.bytesAsHexString((byte[])accountAgeWitness.getHash()) + "\n" + this.signedWitnessService.ownerPubKeyAsString(accountAgeWitness) : "";
        if (this.signedWitnessService.isFilteredWitness(accountAgeWitness)) {
            return SignState.BANNED.addHash(hash);
        }
        if (this.signedWitnessService.isSignedByArbitrator(accountAgeWitness)) {
            return SignState.ARBITRATOR.addHash(hash);
        }
        long accountSignAge = this.getWitnessSignAge(accountAgeWitness, new Date());
        switch (this.getAccountAgeCategory(accountSignAge).ordinal()) {
            case 2: 
            case 3: {
                return SignState.PEER_SIGNER.addHash(hash);
            }
            case 1: {
                return SignState.PEER_INITIAL.addHash(hash).setDaysUntilLimitLifted(30L - TimeUnit.MILLISECONDS.toDays(accountSignAge));
            }
        }
        return SignState.UNSIGNED.addHash(hash);
    }

    public Set<AccountAgeWitness> getOrphanSignedWitnesses() {
        return this.signedWitnessService.getRootSignedWitnessSet(false).stream().map(signedWitness -> this.getWitnessByHash(signedWitness.getAccountAgeWitnessHash()).orElse(null)).filter(Objects::nonNull).collect(Collectors.toSet());
    }

    public void signAndPublishSameNameAccounts() {
        Set<PaymentAccount> signerAccounts = Objects.requireNonNull(this.user.getPaymentAccounts()).stream().filter(account -> account.getOwnerId() != null && this.accountIsSigner(this.getMyWitness(account.getPaymentAccountPayload()))).collect(Collectors.toSet());
        Set unsignedAccounts = this.user.getPaymentAccounts().stream().filter(account -> account.getOwnerId() != null && !this.signedWitnessService.isSignedAccountAgeWitness(this.getMyWitness(account.getPaymentAccountPayload()))).collect(Collectors.toSet());
        signerAccounts.forEach(signer -> unsignedAccounts.forEach(unsigned -> {
            if (signer.getOwnerId().equals(unsigned.getOwnerId())) {
                try {
                    this.signedWitnessService.selfSignAndPublishAccountAgeWitness(this.getMyWitness(unsigned.getPaymentAccountPayload()));
                }
                catch (CryptoException e) {
                    log.warn("Self signing failed, exception {}", (Object)e.toString());
                }
            }
        }));
    }

    public Set<SignedWitness> getUnsignedSignerPubKeys() {
        return this.signedWitnessService.getUnsignedSignerPubKeys();
    }

    public boolean isSignWitnessTrade(Trade trade) {
        Preconditions.checkNotNull((Object)trade, (Object)"trade must not be null");
        Preconditions.checkNotNull((Object)trade.getOffer(), (Object)"offer must not be null");
        PaymentAccountPayload sellerPaymentAccountPayload = trade.getSeller().getPaymentAccountPayload();
        AccountAgeWitness myWitness = this.getMyWitness(sellerPaymentAccountPayload);
        this.getAccountAgeWitnessUtils().witnessDebugLog(trade, myWitness);
        return this.accountIsSigner(myWitness) && !this.peerHasSignedWitness(trade) && this.tradeAmountIsSufficient(trade.getAmount());
    }

    public String getSignInfoFromAccount(PaymentAccount paymentAccount) {
        PublicKey pubKey = this.keyRing.getSignatureKeyPair().getPublic();
        AccountAgeWitness witness = this.getMyWitness(paymentAccount.getPaymentAccountPayload());
        return Utilities.bytesAsHexString((byte[])witness.getHash()) + "," + Utilities.bytesAsHexString((byte[])pubKey.getEncoded());
    }

    public Tuple2<AccountAgeWitness, byte[]> getSignInfoFromString(String signInfo) {
        String[] parts = signInfo.split(",");
        if (parts.length != 2) {
            return null;
        }
        try {
            byte[] accountAgeWitnessHash = Utilities.decodeFromHex((String)parts[0]);
            byte[] pubKeyHash = Utilities.decodeFromHex((String)parts[1]);
            Optional<AccountAgeWitness> accountAgeWitness = this.getWitnessByHash(accountAgeWitnessHash);
            return accountAgeWitness.map(ageWitness -> new Tuple2(ageWitness, (Object)pubKeyHash)).orElse(null);
        }
        catch (Exception e) {
            return null;
        }
    }

    public AccountAgeWitnessUtils getAccountAgeWitnessUtils() {
        return this.accountAgeWitnessUtils;
    }

    public static enum AccountAge {
        UNVERIFIED,
        LESS_ONE_MONTH,
        ONE_TO_TWO_MONTHS,
        TWO_MONTHS_OR_MORE;

    }

    public static enum SignState {
        UNSIGNED(Res.get("offerbook.timeSinceSigning.notSigned")),
        ARBITRATOR(Res.get("offerbook.timeSinceSigning.info.arbitrator")),
        PEER_INITIAL(Res.get("offerbook.timeSinceSigning.info.peer")),
        PEER_LIMIT_LIFTED(Res.get("offerbook.timeSinceSigning.info.peerLimitLifted")),
        PEER_SIGNER(Res.get("offerbook.timeSinceSigning.info.signer")),
        BANNED(Res.get("offerbook.timeSinceSigning.info.banned"));

        private String displayString;
        private String hash = "";
        private long daysUntilLimitLifted = 0L;

        private SignState(String displayString) {
            this.displayString = displayString;
        }

        public SignState addHash(String hash) {
            this.hash = hash;
            return this;
        }

        public SignState setDaysUntilLimitLifted(long days) {
            this.daysUntilLimitLifted = days;
            return this;
        }

        public String getDisplayString() {
            if (!this.hash.isEmpty()) {
                return this.displayString + " " + this.hash;
            }
            return String.format(this.displayString, this.daysUntilLimitLifted);
        }

        public boolean isLimitLifted() {
            return this == PEER_LIMIT_LIFTED || this == PEER_SIGNER || this == ARBITRATOR;
        }
    }
}

