/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.store;

import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.Nullable;
import org.bitcoinj.core.Block;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.ProtocolException;
import org.bitcoinj.core.Sha256Hash;
import org.bitcoinj.core.StoredBlock;
import org.bitcoinj.store.BlockStore;
import org.bitcoinj.store.BlockStoreException;
import org.bitcoinj.store.ChainFileLockedException;
import org.bitcoinj.utils.Threading;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SPVBlockStore
implements BlockStore {
    private static final Logger log = LoggerFactory.getLogger(SPVBlockStore.class);
    public static final int DEFAULT_CAPACITY = 5000;
    public static final String HEADER_MAGIC = "SPVB";
    protected volatile MappedByteBuffer buffer;
    protected final NetworkParameters params;
    protected ReentrantLock lock = Threading.lock("SPVBlockStore");
    protected LinkedHashMap<Sha256Hash, StoredBlock> blockCache = new LinkedHashMap<Sha256Hash, StoredBlock>(){

        @Override
        protected boolean removeEldestEntry(Map.Entry<Sha256Hash, StoredBlock> entry) {
            return this.size() > 2050;
        }
    };
    private static final Object NOT_FOUND_MARKER = new Object();
    protected LinkedHashMap<Sha256Hash, Object> notFoundCache = new LinkedHashMap<Sha256Hash, Object>(){

        @Override
        protected boolean removeEldestEntry(Map.Entry<Sha256Hash, Object> entry) {
            return this.size() > 100;
        }
    };
    protected FileLock fileLock = null;
    protected RandomAccessFile randomAccessFile = null;
    private int fileLength;
    private final String fileAbsolutePath;
    protected StoredBlock lastChainHead = null;
    protected static final int RECORD_SIZE = 128;
    protected static final int FILE_PROLOGUE_BYTES = 1024;

    public SPVBlockStore(NetworkParameters params, File file) throws BlockStoreException {
        this(params, file, 5000, false);
    }

    public SPVBlockStore(NetworkParameters params, File file, int capacity, boolean grow) throws BlockStoreException {
        Preconditions.checkNotNull((Object)file);
        this.fileAbsolutePath = file.getAbsolutePath();
        this.params = (NetworkParameters)Preconditions.checkNotNull((Object)params);
        Preconditions.checkArgument((capacity > 0 ? 1 : 0) != 0);
        try {
            boolean exists = file.exists();
            this.randomAccessFile = new RandomAccessFile(file, "rw");
            this.fileLength = SPVBlockStore.getFileSize(capacity);
            if (!exists) {
                log.info("Creating new SPV block chain file " + file);
                this.randomAccessFile.setLength(this.fileLength);
            } else {
                long currentLength = this.randomAccessFile.length();
                if (currentLength != (long)this.fileLength) {
                    if ((currentLength - 1024L) % 128L != 0L) {
                        throw new BlockStoreException("File size on disk indicates this is not a block store: " + currentLength);
                    }
                    if (!grow) {
                        throw new BlockStoreException("File size on disk does not match expected size: " + currentLength + " vs " + this.fileLength);
                    }
                    if ((long)this.fileLength < this.randomAccessFile.length()) {
                        throw new BlockStoreException("Shrinking is unsupported: " + currentLength + " vs " + this.fileLength);
                    }
                    this.randomAccessFile.setLength(this.fileLength);
                }
            }
            FileChannel channel = this.randomAccessFile.getChannel();
            this.fileLock = channel.tryLock();
            if (this.fileLock == null) {
                throw new ChainFileLockedException("Store file is already locked by another process");
            }
            this.buffer = channel.map(FileChannel.MapMode.READ_WRITE, 0L, this.fileLength);
            if (exists) {
                byte[] header = new byte[4];
                this.buffer.get(header);
                if (!new String(header, StandardCharsets.US_ASCII).equals(HEADER_MAGIC)) {
                    throw new BlockStoreException("Header bytes do not equal SPVB");
                }
            } else {
                this.initNewStore(params);
            }
        }
        catch (Exception e) {
            try {
                if (this.randomAccessFile != null) {
                    this.randomAccessFile.close();
                }
            }
            catch (IOException e2) {
                throw new BlockStoreException(e2);
            }
            throw new BlockStoreException(e);
        }
    }

    private void initNewStore(NetworkParameters params) throws Exception {
        byte[] header = HEADER_MAGIC.getBytes("US-ASCII");
        this.buffer.put(header);
        this.lock.lock();
        try {
            this.setRingCursor(this.buffer, 1024);
        }
        finally {
            this.lock.unlock();
        }
        Block genesis = params.getGenesisBlock().cloneAsHeader();
        StoredBlock storedGenesis = new StoredBlock(genesis, genesis.getWork(), 0);
        this.put(storedGenesis);
        this.setChainHead(storedGenesis);
    }

    public static final int getFileSize(int capacity) {
        return 128 * capacity + 1024;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void put(StoredBlock block) throws BlockStoreException {
        MappedByteBuffer buffer = this.buffer;
        if (buffer == null) {
            throw new BlockStoreException("Store closed");
        }
        this.lock.lock();
        try {
            int cursor = this.getRingCursor(buffer);
            if (cursor == this.fileLength) {
                cursor = 1024;
            }
            buffer.position(cursor);
            Sha256Hash hash = block.getHeader().getHash();
            this.notFoundCache.remove(hash);
            buffer.put(hash.getBytes());
            block.serializeCompact(buffer);
            this.setRingCursor(buffer, buffer.position());
            this.blockCache.put(hash, block);
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    @Nullable
    public StoredBlock get(Sha256Hash hash) throws BlockStoreException {
        MappedByteBuffer buffer = this.buffer;
        if (buffer == null) {
            throw new BlockStoreException("Store closed");
        }
        this.lock.lock();
        try {
            int cursor;
            StoredBlock cacheHit = this.blockCache.get(hash);
            if (cacheHit != null) {
                StoredBlock storedBlock = cacheHit;
                return storedBlock;
            }
            if (this.notFoundCache.get(hash) != null) {
                StoredBlock storedBlock = null;
                return storedBlock;
            }
            int startingPoint = cursor = this.getRingCursor(buffer);
            byte[] targetHashBytes = hash.getBytes();
            byte[] scratch = new byte[32];
            do {
                if ((cursor -= 128) < 1024) {
                    cursor = this.fileLength - 128;
                }
                buffer.position(cursor);
                buffer.get(scratch);
                if (!Arrays.equals(scratch, targetHashBytes)) continue;
                StoredBlock storedBlock = StoredBlock.deserializeCompact(this.params, buffer);
                this.blockCache.put(hash, storedBlock);
                StoredBlock storedBlock2 = storedBlock;
                return storedBlock2;
            } while (cursor != startingPoint);
            this.notFoundCache.put(hash, NOT_FOUND_MARKER);
            StoredBlock storedBlock = null;
            return storedBlock;
        }
        catch (ProtocolException e) {
            throw new RuntimeException(e);
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public StoredBlock getChainHead() throws BlockStoreException {
        MappedByteBuffer buffer = this.buffer;
        if (buffer == null) {
            throw new BlockStoreException("Store closed");
        }
        this.lock.lock();
        try {
            if (this.lastChainHead == null) {
                byte[] headHash = new byte[32];
                buffer.position(8);
                buffer.get(headHash);
                Sha256Hash hash = Sha256Hash.wrap(headHash);
                StoredBlock block = this.get(hash);
                if (block == null) {
                    throw new BlockStoreException("Corrupted block store: could not find chain head: " + hash + "\nFile path: " + this.fileAbsolutePath);
                }
                this.lastChainHead = block;
            }
            StoredBlock storedBlock = this.lastChainHead;
            return storedBlock;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void setChainHead(StoredBlock chainHead) throws BlockStoreException {
        MappedByteBuffer buffer = this.buffer;
        if (buffer == null) {
            throw new BlockStoreException("Store closed");
        }
        this.lock.lock();
        try {
            this.lastChainHead = chainHead;
            byte[] headHash = chainHead.getHeader().getHash().getBytes();
            buffer.position(8);
            buffer.put(headHash);
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void close() throws BlockStoreException {
        try {
            this.buffer.force();
            this.buffer = null;
            this.randomAccessFile.close();
            this.blockCache.clear();
        }
        catch (IOException e) {
            throw new BlockStoreException(e);
        }
    }

    @Override
    public NetworkParameters getParams() {
        return this.params;
    }

    private int getRingCursor(ByteBuffer buffer) {
        int c = buffer.getInt(4);
        Preconditions.checkState((c >= 1024 ? 1 : 0) != 0, (Object)"Integer overflow");
        return c;
    }

    private void setRingCursor(ByteBuffer buffer, int newCursor) {
        Preconditions.checkArgument((newCursor >= 0 ? 1 : 0) != 0);
        buffer.putInt(4, newCursor);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clear() throws Exception {
        this.lock.lock();
        try {
            this.blockCache.clear();
            this.notFoundCache.clear();
            this.buffer.position(0);
            long fileLength = this.randomAccessFile.length();
            int i = 0;
            while ((long)i < fileLength) {
                this.buffer.put((byte)0);
                ++i;
            }
            this.buffer.position(0);
            this.initNewStore(this.params);
        }
        finally {
            this.lock.unlock();
        }
    }
}

